from typing import Union, Tuple, List, Dict, Any,Optional

def build_planner_prompt() -> str:
    return """
        You are a planner in an image quality assessment (IQA) system. Your task is to analyze the user's query and generate a structured plan for downstream assessment.

        Return a valid JSON object in the following format:
        {
        "query_type": "IQA" or "others",
        "object_names": ["<object1>", "<object2>", ...] or null,
        "distortions": {
            "<object_name or global>": ["<distortion_1>", "<distortion_2>"]
        } or null,
        "reference_type": "Full-Reference" or "No-Reference",
        "distortion_source": "explicit" or "inferred",
        "required_tool": ["<tool_name_1>", "<tool_name_2>", ...] or null,
        "plan": {
            "distortion_detection": true or false,
            "distortion_analysis": true or false,
            "tool_selection": true or false,
            "tool_execute": true or false
        }
        }

        Instructions:

        1. Reference Type:
        - If both distorted and reference images are present, set "reference_type" to "Full-Reference".
        - Otherwise, set to "No-Reference".

        2. Query Type:
        - If the question focuses on visual distortions (e.g., noise, blur, lighting, sharpness), set "query_type": "IQA".
        - If the question relates to emotion, style, beauty, others, or visual appeal, set "query_type": "others".

        3. Distortion Handling (for IQA queries only):
        - If the query clearly mentions distortion types or attributes, such as blur, noise, sharpness, color, brightness, etc:
            - Set "distortion_source": "explicit"
            - Fill the "distortions" field accordingly
            - Set "distortion_detection": false
        - If the query does not explicitly mention distortions:
            - Set "distortion_source": "inferred"
            - Set "distortions": null
            - Set "distortion_detection": true

        4. Distortion Analysis:
        - If "query_type" is "IQA", always set "distortion_analysis": true
        - If "query_type" is "others", set "distortion_analysis": false

        5. Tool and Execution Plan (for "IQA" query type only):
        - If the query refers to specific regions or objects (object-level):
            - Set "object_names": list of regions
            - Set "required_tool": null
            - Set "tool_selection": false
            - Set "tool_execute": false
        - If the query is about the whole image (global-level):
            - If no specific tool is mentioned:
                - Set "required_tool": null
                - Set "tool_selection": true
                - Set "tool_execute": true
            - If specific tools are mentioned:
                - Set "required_tool": list of tools
                - Set "tool_selection": false
                - Set "tool_execute": true

        6. Generic queries that do not imply specific distortions include:
        - "How is the perceptual quality of the image?"
        - "Does the image look good?"
        - "Evaluate the quality of this picture."

        Only return valid JSON. Do not include explanations, markdown, or extra text.
    """

def build_distortion_detection_prompt(object_level: bool,user_question: str) -> str:
    # distortion_types = ['Brightness', 'Colorfulness', 'Contrast', 'Noisiness', 'Sharpness']
    distortion_types = ['Blurs', 'Color distortions', 'Compression', 'Noise', 'Brightness change', 'Spatial distortions', 'Sharpness and contrast']
    region_instr = (
        "Focus only on the specified regions. Compare each region with the reference image if provided."
        if object_level else
        "Analyze the entire distorted image. Compare it with the reference image if provided."
    )

    prompt = f"""
        User question: "{user_question}"
        
        You are a distortion analysis expert. 
        
        Based on the following **user question**, identify all possible distortions need to be focused on to properly address the user's intent.

        Select the distortions from the following list: {distortion_types}
        
        Output format (valid JSON only):

        {{
        "distortions": {{
            "<object_name or 'global'>": ["<distortion_type_1>", "<distortion_type_2>"]
        }}
        }}

        Instructions:
        1. {region_instr}
        2. Analyze each of the distortion types listed above.
        3. Only return valid JSON. Do not include extra text or markdown.
        """
    return prompt

def build_distortion_analysis_prompt_multi_object(
    distortion_dict: Dict[str, List[str]],
    has_reference: bool = False,
    user_question: Optional[str] = ""
) -> str:
    region_blocks = []
    for obj_name, dist_list in distortion_dict.items():
        dist_str = ", ".join(dist_list)
        region_blocks.append(f"- Region: {obj_name}\n  Distortions to analyze: {dist_str}")
    full_region_description = "\n\n".join(region_blocks)

    image_line = (
        "You are given a distorted image and a reference image. "
        "Compare the two to assess the distortion impact."
        if has_reference else
        "You are given only a distorted image. Analyze the distortions based on visible visual cues."
    )

    prompt = f"""
You are a distortion analysis expert. Your task is to assess the **severity and visual impact** of various distortion types for different regions of an image.

User question: "{user_question}"

{image_line}

The image contains multiple distinct regions, and each region is associated with its own set of distortion types.

Please analyze the following regions independently:

{full_region_description}

Return your output as valid JSON in the format:

{{
  "distortion_analysis": {{
    "region_name_1": [
      {{
        "type": "<distortion_name>",
        "severity": "<none/slight/moderate/severe/extreme>",
        "explanation": "<brief visual impact description>"
      }}
    ],
    ...
  }}
}}

Instructions:
1. Base your analysis on the listed distortion types and consider the user question.
2. Use "none" if a distortion is barely or not visible.
3. In the `explanation` field, briefly describe **where** the distortion appears (e.g., background, face, edges) and **how** it affects the perceived image quality (e.g., reduces sharpness, causes color imbalance, introduces blocky artifacts). Keep the explanation concise and focused on visible visual impact. **Do not identify specific people.**
4. Do not mix results between regions.
5. Only return valid JSON. No explanations, markdown, or comments.
""".strip()
    return prompt

def build_required_tool_prompt(tool_name:str) -> str:
    prompt = f"""
    You are a tool executor. Your task is to execute the {tool_name} required by user and return the score.

    Return your output in the following JSON format:
    {{
        "quality_score": <float between 1 and 5>,
        }}

    Only return valid JSON. Do not include extra text and markdown.
    """
    return prompt
def build_summary_choice_prompt_demo8(choices: List[str], user_question: str) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)

    return f"""
You are a visual quality assessment assistant.

User question: "{user_question}"

Your task is to select the most appropriate answer to the user's question. You are given:
- Distortion analysis: descriptions of severity and visual impact for each distortion type
- Fused scores (if provided): scores for each distortion type
- Overall score (if provided): the image's overall perceptual quality
- Image content (if needed)

Decision process:
1. First, understand what kind of visual information is needed to answer the user's question.
2. Check if the provided distortion analysis or tool response already contains the required information.
3. If the provided information is sufficient, use it to answer.
4. If the information is unclear or insufficient, analyze the image directly to determine the best answer.

Answer must be one of the following choices: {letter_str}

Return your answer strictly in the JSON format shown below.
{{
  "final_response": "<one of the above letters>",
  "reason": "<brief explanation, based on either distortion analysis, tool response, or direct visual observation>"
}}
If you cannot follow the format, return your answer in plain text instead.
"""

def build_summary_choice_prompt(choices: List[str], user_question: str) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)

    return f"""
You are a visual quality assessment assistant.

User question: "{user_question}"

Your task is to select the most appropriate answer to the user's question. You are given:
- Distortion analysis (severity and visual impact of listed distortions)
- Tool response (overall quality scores from IQA models)
- Image content

Decision process:
1. First, understand what kind of visual information is needed to answer the user's question.
2. Check if the provided distortion analysis or tool response already contains the required information.
3. If the provided information is sufficient, use it to answer.
4. If the information is unclear or insufficient, analyze the image directly to determine the best answer.

Answer must be one of the following choices: {letter_str}

Return your answer strictly in the JSON format shown below.
{{
  "final_response": "<one of the above letters>",
  "reason": "<brief explanation, based on either distortion analysis, tool response, or direct visual observation>"
}}
If you cannot follow the format, return your answer in plain text instead.
"""

def build_summary_others_prompt(choices: List[str]) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)
    return f"""
    You are an assistant for others and emotional image assessment.

    Your task is to answer the user's question by directly analyzing the given image.

    Carefully examine the image and choose the option that best answers the question, based on what you see.

    The answer must be one of the following choices: {letter_str}.

    Return your answer strictly in this JSON format:
    {{
        "final_response": "<one of the above letters>",
        "reason": "<brief explanation of why this choice was selected>"
    }}

    Only return the JSON object. Do not include extra text or formatting.
    """

def build_summary_nochoice_prompt() -> str:
    prompt = """
    You are an image quality assessor.

    Given the question and the analysis (tool scores, distortion analysis), assess the image quality.

    You must select **one single answer** from the following:
    A. Excellent
    B. Good
    C. Fair
    D. Poor
    E. Bad

    Return your answer using ONLY the letter A, B, C, D, or E — no extra explanation or formatting.
    """
    return prompt


def build_tool_prompt() -> str:
    prompt = """
    You are a tool executor.
    
    Your task is to assign the most appropriate IQA tool to each visual distortion type, based on the descriptions of the tools.

    Return your output in the following JSON format:
    {
      "selected_tools": {
        "<object_name or 'global'>": {
          "<distortion_name_1>": "<tool_name>",
          "<distortion_name_2>": "<tool_name>"}
        }
    }

    Instructions:
    1. For each distortion, choose the tool whose description suggests it performs best for that type of distortion. If no exact match exists, select the most semantically relevant one.
    2. Tool names must exactly match the provided names. Do NOT add extra prefixes like "functions.".
    3. Return only valid JSON. Do not include explanations, markdown, or extra text.
    """
    return prompt

def build_summary_choice_prompt1(choices: List[str],user_question: str) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)

    return f"""
        You are a visual quality assessment assistant.

        User question: "{user_question}"

        Your task is to decide whether the provided information is sufficient to answer the user's question.

        Available inputs:
        - Distortion analysis: severity and visual impact of distortions.
        - Tool response: overall quality scores from IQA models (if provided).

        Decision process:
        1. Consider the distortion analysis and tool scores (if provided) together.
        2. If they clearly support one answer and sufficient to make a decision, return it with reflection=false and answer the user's question.
        3. If they are ambiguous or contradictory, set reflection=true and do not provide an answer.

        Return your output in this JSON format:
        {{
        "final_response": "<one of the letters: {letter_str}> or null",
        "reason": "<brief explanation> or null",
        "reflection": true or false
        }}

        Do not reference the image directly. Only return the JSON object.
        """.strip()
